#include "BMPWriterStrategy.h"
#include <stdexcept>
#include <cstring>
#include <vector>
#include <string>
#include <cstdint>
#include "ImageData.h"

#pragma pack(push, 1)
struct BMPFileHeader
{
    uint16_t bfType;      // "BM"
    uint32_t bfSize;      // File size
    uint16_t bfReserved1; // Reserved
    uint16_t bfReserved2; // Reserved
    uint32_t bfOffBits;   // Offset to pixel data
};

struct BMPInfoHeader
{
    uint32_t biSize;         // Header size
    int32_t biWidth;         // Image width
    int32_t biHeight;        // Image height
    uint16_t biPlanes;       // Color planes (must be 1)
    uint16_t biBitCount;     // Bits per pixel
    uint32_t biCompression;  // Compression type
    uint32_t biSizeImage;    // Image size
    int32_t biXPelsPerMeter; // Horizontal resolution
    int32_t biYPelsPerMeter; // Vertical resolution
    uint32_t biClrUsed;      // Colors used
    uint32_t biClrImportant; // Important colors
};
#pragma pack(pop)

std::vector<uint8_t> BMPWriterStrategy::write(const ImageData &imageData) const
{
    int16_t width = imageData.width;
    int16_t height = imageData.height;

    // Calculate row size with padding (must be multiple of 4)
    uint32_t rowSize = ((width * 3 + 3) / 4) * 4;
    uint32_t pixelDataSize = rowSize * height;

    // Prepare headers
    BMPFileHeader fileHeader;
    fileHeader.bfType = 0x4D42; // "BM"
    fileHeader.bfSize = sizeof(BMPFileHeader) + sizeof(BMPInfoHeader) + pixelDataSize;
    fileHeader.bfReserved1 = 0;
    fileHeader.bfReserved2 = 0;
    fileHeader.bfOffBits = sizeof(BMPFileHeader) + sizeof(BMPInfoHeader);

    BMPInfoHeader infoHeader;
    infoHeader.biSize = sizeof(BMPInfoHeader);
    infoHeader.biWidth = width;
    infoHeader.biHeight = height;
    infoHeader.biPlanes = 1;
    infoHeader.biBitCount = 24;   // 24-bit RGB
    infoHeader.biCompression = 0; // No compression
    infoHeader.biSizeImage = pixelDataSize;
    infoHeader.biXPelsPerMeter = 2835; // 72 DPI
    infoHeader.biYPelsPerMeter = 2835;
    infoHeader.biClrUsed = 0;
    infoHeader.biClrImportant = 0;

    // Create output vector
    std::vector<uint8_t> bmpData;
    bmpData.reserve(fileHeader.bfSize);

    // Write file header to vector
    const uint8_t* fileHeaderPtr = reinterpret_cast<const uint8_t*>(&fileHeader);
    bmpData.insert(bmpData.end(), fileHeaderPtr, fileHeaderPtr + sizeof(BMPFileHeader));

    // Write info header to vector
    const uint8_t* infoHeaderPtr = reinterpret_cast<const uint8_t*>(&infoHeader);
    bmpData.insert(bmpData.end(), infoHeaderPtr, infoHeaderPtr + sizeof(BMPInfoHeader));

    // Write pixel data (BMP stores bottom-to-top, BGR format)
    std::vector<uint8_t> rowData(rowSize, 0);

    for (int16_t y = height - 1; y >= 0; --y)
    {
        for (int16_t x = 0; x < width; ++x)
        {
            uint32_t pixelIndex = y * width + x;

            // BMP uses BGR format
            rowData[x * 3 + 0] = imageData.B[pixelIndex];
            rowData[x * 3 + 1] = imageData.G[pixelIndex];
            rowData[x * 3 + 2] = imageData.R[pixelIndex];
        }

        bmpData.insert(bmpData.end(), rowData.begin(), rowData.end());
    }

    return bmpData;
}

