import struct

class ANAFile:
    def __init__(self, header, rle_data, width, height):
        self.header = header          # bytes
        self.rle_data = rle_data      # bytes
        self.width = width
        self.height = height

    def write_to_file(self, filename):
        with open(filename, "wb") as f:
            f.write(self.header)
            f.write(self.rle_data)

        print(f"Successfully created ANA file: {filename} ({len(self.header) + len(self.rle_data)} bytes).")


class ANAFileBuilder:
    def __init__(self):
        self._width = 0
        self._height = 0
        self._rle_data = bytearray()

    def set_dimensions(self, width, height):
        self._width = width
        self._height = height
        return self

    def add_rle_chunk(self, count, r, g, b):
        self._rle_data += bytes([count, r, g, b])
        return self

    def build(self):
        if self._width == 0:
            raise RuntimeError("Dimensions must be set before building.")

        header = bytearray()

        # Signature "ANA\0"
        header += b'\x41\x4E\x41\x00'

        # width, height (little endian uint16)
        header += struct.pack('<H', self._width)
        header += struct.pack('<H', self._height)

        return ANAFile(bytes(header), bytes(self._rle_data), self._width, self._height)


if __name__ == "__main__":
    print("--- ANA File Builder Demonstration ---")

    try:
        my_file = (
            ANAFileBuilder()
            .set_dimensions(3, 3) # define width and height
            .add_rle_chunk(3, 0xFF, 0x00, 0x00) # N, R, G, B
            .add_rle_chunk(1, 0x00, 0xFF, 0x00)
            .add_rle_chunk(2, 0x00, 0x00, 0xFF)
            .add_rle_chunk(3, 0xFF, 0xFF, 0xFF)
            .build()
        )

        my_file.write_to_file("sample.ana")

    except Exception as e:
        print(f"Fatal Error: {e}")
